// RatPlugin.cpp starts
/*
	CGP, Created, 12/24/00: Needed this to separate program code for rat API from rat
		implementations. The rat API became more complicated with having the rat
		block when he comes off the maze instead of having the rat explicity wait for
		messages.
*/

// Amount to wait between moves of rat.
// See comment with usage below.
#define		WAIT_AMOUNT			0.30

#include <Message.h>
#include <stdio.h>
#include "RatEnvMessage.h"
#include "Debug.h"
#include "PortMessage.h"
#include "DatabaseMessage.h"
#include "RatPlugin.h"
#include "EnvironmentConstants.h"
#include "View.h"

// Globals in Rat Plugin; need a copy of this here, but only in the maze simulator
#ifndef MAZE_SIMULATOR
DebugServer *GLOB_debug = NULL;
#endif

RatPlugin::RatPlugin(PortMessage *ratPort, DatabaseMessage *dbMsg, RatEnvMessage *ratEnvMsg,
	DebugServer *bugServer) {
	
	ratOnMaze = false;
   	this->envMsg = ratEnvMsg;
   	this->ratPort = ratPort;
   	this->dbMsg = dbMsg;
   	stepMode = false;
   	GLOB_debug = bugServer;
};

RatPlugin::~RatPlugin()
{
}

bool RatPlugin::VersionOK() {
	if ((envMsg->Ver() == RAT_ENV_MESSAGE_API_VER) &&
		(PLUGIN_API_Version() == RAT_API_VER)) {
		return true;
	} else {
		return false;
	}
}

void RatPlugin::RAT_Wait(float secs) {
	snooze((int)(secs*1000000.0)); // BeOS specific call!
}

void RatPlugin::RAT_AddDatabaseRecord(BMessage *record) {
	dbMsg->AddRecord(record);	
}

/*
BMessage *GetIncomingMessage() {
	return ratPort->GetMessage();
}

bool IsIncomingMessageReady() {
	return ratPort->IsMessageReady();
}

BMessage *PeekAtIncomingMessage() {
	return ratPort->PeekMessage();
}

void FlushIncomingMessages() {
	ratPort->Flush();
}
*/

// Function to process incoming messages from the experimenter, updating the
// "ratOnMaze" variable. If this variable is true after processing messages,
// rat proceeds. Otherwise, he blocks waiting for the variable to be updated
// to be true.
void RatPlugin::BlockUntilRatOnMaze()
{
	char buffer[100];
	float duration;
	
	// Stay in the loop while the rat is not on the maze or there are messages
	// ready to be processed.	
	while ((! ratOnMaze) || ratPort->IsMessageReady()) {
		BMessage *msg = ratPort->GetMessage(); // blocking message receive
		
		sprintf(buffer, "%d", (int) msg->what);
		Debug('n', "RatPlugin::BlockUntilRatOnMaze: Received message= ", buffer);
		
		switch (msg->what) {
		// This message is from the GUI. The rat is now in run mode (continuous steps).
		case RATMSG_RUN_MODE:
			stepMode = false;
			break;
		
		// This message is from the GUI. The rat is now in single step mode.
		case RATMSG_STEP_MODE:
			stepMode = true;
			break;
		
		// Run a trial for the rat if we are in run mode. Someone needs to tell the rat to stop
		// afterwards. This message should come from the experimenter so that he can tell
		// the rat to stop afterwards.
		// Run a step if we are in step mode.
		case RATMSG_SINGLE_STEP:
		case RATMSG_GO:
			if (stepMode) {
				Debug('n', "RatPlugin::BlockUntilRatOnMaze: Step mode, running a step");
				ratOnMaze = true;
			} else {
				Debug('n', "RatPlugin::BlockUntilRatOnMaze: Run mode, running rat");
				// Not all GO messages will have trialType info, but some will.
				if (msg->FindInt16("trialType", &trialType) != B_OK) {
					Debug('n', "RatPlugin::BlockUntilRatOnMaze: Could not get trial type info");
				} else {
					trialType = -1;
				}
				ratOnMaze = true;
			}
			break;
			
		case RATMSG_STOP:
			ratOnMaze = false;
			duration = 0.0;
			if (msg->FindFloat("duration", &duration) != B_OK) {
				Debug('n', "RatPlugin::BlockUntilRatOnMaze: Could not get duration info");
			}
			PLUGIN_TakeOffMaze(duration);
			break;
			
		default:
			sprintf(buffer, "%d", (int) msg->what);
			Debug('n', "RatPlugin::BlockUntilRatOnMaze: Invalid message= ", buffer);
			break;
		}
		
		delete msg;
	}
	// We've gotten out of the loop. This must mean that ratOnMaze is true
	// and there are no more messages waiting. So, we can return from this function.
}

void RatPlugin::Release()
{
	if (stepMode) {
		// If we are in single step mode, take rat off maze after each action/perception operation
		// he carries out.
		ratOnMaze = false;
	}
}

int RatPlugin::RAT_GetCurrentTrialType()
{
	return (int) trialType;
}

void RatPlugin::RAT_Rotate(int degrees) {
	BlockUntilRatOnMaze();
	Debug('n', "RatPlugin::Rotate");
	envMsg->Rotate(degrees);
	Release();
}

bool RatPlugin::RAT_IsOpen() {
	BlockUntilRatOnMaze();
	Debug('n', "RatPlugin::IsOpen");
	bool result = envMsg->IsOpen();
	Release();
	return result;
}

bool RatPlugin::RAT_Move() {
	BlockUntilRatOnMaze();
	Debug('n', "RatPlugin::Move");
	bool result = envMsg->Move();
	/* The following wait is a bit of a hack right now. The maze graphical display
	 	is strongly dependent upon having this wait here. If you take this out, the maze
	 	display updates too rapidly. This should be fixed so that we don't need the wait
	 	here, and so that the maze updates itself appropriately (less quickly).
	*/
	RAT_Wait(WAIT_AMOUNT);
	Release();
	return result;
}

bool RatPlugin::RAT_FoodHere() {
	BlockUntilRatOnMaze();
	Debug('n', "RatPlugin::FoodHere");
	bool result = envMsg->FoodHere();
	Release();
	return result;
}

bool RatPlugin::RAT_EatFood() {
	BlockUntilRatOnMaze();
	Debug('n', "RatPlugin::EatFood");
	bool result = envMsg->EatFood();
	Release();
	return result;
}

float *RatPlugin::RAT_Look() {	
	BlockUntilRatOnMaze();
	Debug('n', "RatPlugin::Look");
	float *result = envMsg->Look();
	Release();
	return result;
}

void RatPlugin::RAT_UserMessage(char *msg)
{
	DebugGUI(msg);
}

// We can restrict ourself to arms that are open because we are trying to figure
// out which of the two open arms is the one we are choosing.
bool RatPlugin::RAT_RotateUntilView(float *view)
{
	// In this loop, don't increment/rotate more than by 1 degree because if we
	// increment by a larger number there is a chance we'll miss matching
	// exactly with the view we looked at before.
	// This shouldn't be too inefficient as IsOpen won't return true that many
	// times.
	for (int i=0; i < 360; i++) {
		if (RAT_IsOpen()) {
			// Look out from current rotation & see if we're heading towards view
	  		float *currView = RAT_Look();
	  		int same = ViewsEqual(view, currView);
	  		delete currView;
	  		if (same) return true;
	  	}
	  	
	  	// Door to arm was not open or currView is not same as view.
	  	// So rotate more.
	  	RAT_Rotate(1);
	}
	
	return false;
}

// RatPlugin.cpp ends
